  ' -----------------------------------------------------------
  ' OLED MODULE TEST prog2.bas
  ' A program for Micromite (V.1, V.2 or Plus Explore 64) to
  ' connect to a 1.3 inch OLED display module based on the
  ' SH1106 or SSD1306 controller/driver chips.
  '
  ' Written by Jim Rowe for Silicon Chip, with much appreciated
  ' help from Tim Blythman.
  ' Last revision 16/3/2023 at 7:35am
  '
  ' Notes:
  ' 1. Communication with the module is via the Micromite's
  ' I2C port. For the original Micromite this uses pin 17
  ' for SCL and pin 18 for SDA; while for the Micromite Plus
  ' Explore 64 pin 44 is used for SCL and pin 43 for SDA.
  ' 2. The SH1106/SSD1306 chip's 8-bit I2C address is specified as 78h,
  ' although this can be changed to 7Ah by removing the 4.7k#
  ' resistor connecting its DC pin (15) to ground, and instead using
  ' a 4.k# resistor to connect the pin to +3.3V.
  ' 3. MMBasic's I2C function expects a 7-bit I2C slave address since
  ' it adds the lsb read/write bit automatically; R = 1, W = 0.
  ' as a result we need to give the SH1106/SSD1306 an I2C address
  ' of either 3Ch or 3Dh.
  ' 4. The 1.3" OLED display provides a resolution of 132 x 64 pixels,
  ' but is normally used to display a matrix of 128 x 64 pixels with
  ' each pixel either off (black) or on (either white or blue).
  ' 5. The SH1106 controller chip includes a 132 x 64 bit SRAM, into
  ' which the data to be displayed is stored after receipt from the
  ' MCU via the I2C interface.
  ' 6. The SH1106 chip also receives commands from the MCU, via the
  ' same I2C interface. The commands are stored and decoded to perform
  ' the required tasks.
  ' 7. In this program we are using the OLED driver chip's Page
  ' Addressing mode, which seems to work for both the SH1106 and SSD1106
  '
  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM STRING PStr$(128)  ' string to store one page of SH1106/SSD1306
  ' display data (128 bytes)
  DIM STRING SendData$
  DIM STRING TStr$
  DIM INTEGER Cntr        ' loop counter
  
  I2C OPEN 400, 200       ' enable the Micromite I2C module in master mode
  ' (speed = 400kHz, timeout = 200 milliseconds
  OLEDcmd(&HAF)  ' turn OLED on
  OLEDcmd(&H22)  ' set OLED for Page addressing mode
  ContrastSet    ' then go set the OLED contrast (brightness)
  ScrnClear      ' then go clear the complete OLED screen
  OLEDcmd(&HB0)  ' now set for Page 0
  OLEDcmd(&H02)  ' then set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  AssemblePage0  ' then go assemble the page image
  OLEDsend       ' and send the page of data to the OLED RAM
  OLEDcmd(&HB2)  ' now set for Page 2
  OLEDcmd(&H02)  ' then set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  AssemblePage2  ' then assemble page 2 image
  OLEDsend       ' and send it to the OLED
  OLEDcmd(&HB7)  ' now set for Page 7
  OLEDcmd(&H02)  ' then set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  OLEDsend       ' and send Page 2 to the OLED again, but to Page 7
  OLEDcmd(&HB3)  ' now set for Page 3
  OLEDcmd(&H02)  ' then set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  AssemblePage3  ' then assemble page 3 image
  OLEDsend       ' and send it to the OLED 
  OLEDcmd(&HB4)  ' now set for Page 4
  OLEDcmd(&H02)  ' then set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  OLEDsend       ' and send Page 2 to the OLED again, but to Page 4
  OLEDcmd(&HB6)  ' now set for Page 6
  OLEDcmd(&H02)  ' then set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  OLEDsend       ' and send Page 2 to the OLED again, but to Page 6
  OLEDcmd(&HB5)  ' now set for Page 5
  OLEDcmd(&H02)  ' then set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  AssemblePage5  ' and assemble page 5 image
  OLEDsend       ' and send it to the OLED Page 5     
  OLEDcmd(&HA4)  ' finally send command to start up OLED
END
  ' ---------------------------------------------------------------------
  
  ' subroutine to assemble the page image to be displayed
SUB AssemblePage0
  PStr$(0)="" ' clear string for a clean start
  PStr$(0)=PStr$(0)+CHR$(&H7F)+CHR$(&H08)+CHR$(&H08)+CHR$(&H7F)+CHR$(&H00) 'H
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H54)+CHR$(&H54)+CHR$(&H18)+CHR$(&H00) 'e
  PStr$(0)=PStr$(0)+CHR$(&H7F)+CHR$(&H40)+CHR$(&H00) 'l
  PStr$(0)=PStr$(0)+CHR$(&H7F)+CHR$(&H40)+CHR$(&H00) 'l
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H38)+CHR$(&H00) 'o
  PStr$(0)=PStr$(0)+STRING$(3,0) 'space
  PStr$(0)=PStr$(0)+CHR$(&H3F)+CHR$(&H40)+CHR$(&H3F)+CHR$(&H40)+CHR$(&H3F)+CHR$(&H00) 'W
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H38)+CHR$(&H00) 'o
  PStr$(0)=PStr$(0)+CHR$(&H7C)+CHR$(&H08)+CHR$(&H04)+CHR$(&H08)+CHR$(&H00) 'r
  PStr$(0)=PStr$(0)+CHR$(&H7F)+CHR$(&H40)+CHR$(&H00) 'l
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H38)+CHR$(&H7F)+CHR$(&H00) 'd
  PStr$(0)=PStr$(0)+CHR$(&H5F)+CHR$(&H00) '!
  PStr$(0)=PStr$(0)+STRING$(3,0) ' space
  PStr$(0)=PStr$(0)+CHR$(&H26)+CHR$(&H49)+CHR$(&H49)+CHR$(&H32)+CHR$(&H00) 'S
  PStr$(0)=PStr$(0)+CHR$(&H7D)+CHR$(&H40)+CHR$(&H00) 'i
  PStr$(0)=PStr$(0)+CHR$(&H7F)+CHR$(&H40)+CHR$(&H00) 'l
  PStr$(0)=PStr$(0)+CHR$(&H7D)+CHR$(&H40)+CHR$(&H00) 'i
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H08)+CHR$(&H00) 'c
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H38)+CHR$(&H00) 'o
  PStr$(0)=PStr$(0)+CHR$(&H7C)+CHR$(&H08)+CHR$(&H04)+CHR$(&H78)+CHR$(&H00) 'n
  PStr$(0)=PStr$(0)+STRING$(3,0) ' space
  PStr$(0)=PStr$(0)+CHR$(&H3E)+CHR$(&H41)+CHR$(&H41)+CHR$(&H22)+CHR$(&H00) 'C
  PStr$(0)=PStr$(0)+CHR$(&H7F)+CHR$(&H08)+CHR$(&H04)+CHR$(&H78)+CHR$(&H00) 'h
  PStr$(0)=PStr$(0)+CHR$(&H7D)+CHR$(&H40)+CHR$(&H00) 'i
  PStr$(0)=PStr$(0)+CHR$(&HFC)+CHR$(&H22)+CHR$(&H22)+CHR$(&H1C)+CHR$(&H00) 'p
  PStr$(0)=PStr$(0)+STRING$(3,0) ' space
  PStr$(0)=PStr$(0)+CHR$(&H7F)+CHR$(&H04)+CHR$(&H04)+CHR$(&H78)+CHR$(&H00) 'h
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H54)+CHR$(&H54)+CHR$(&H18)+CHR$(&H00) 'e
  PStr$(0)=PStr$(0)+CHR$(&H7C)+CHR$(&H08)+CHR$(&H04)+CHR$(&H08)+CHR$(&H00) 'r
  PStr$(0)=PStr$(0)+CHR$(&H38)+CHR$(&H54)+CHR$(&H54)+CHR$(&H18)+CHR$(&H00) 'e
  
END SUB
  ' ------------------------------------------------------------------
  ' subroutine to wipe the OLED screen
SUB ScrnClear
  OLEDcmd(&HB0) ' set for Page 0
  Pageclear     ' and go clear it
  OLEDsend
  OLEDcmd(&HB1) ' then set for Page 1
  Pageclear     ' and clear it
  OLEDsend
  OLEDcmd(&HB2) ' and keep going
  Pageclear
  OLEDsend
  OLEDcmd(&HB3)
  Pageclear
  OLEDsend
  OLEDcmd(&HB4)
  Pageclear
  OLEDsend
  OLEDcmd(&HB5)
  Pageclear
  OLEDsend
  OLEDcmd(&HB6)
  Pageclear
  OLEDsend
  OLEDcmd(&HB7) ' last Page
  Pageclear
  OLEDsend
END SUB
  ' ------------------------------------------------------------------
  ' subroutine to send all or some of the Page image to the OLED
SUB OLEDsend
  SendData$ = CHR$(&H40) + PStr$(0)
  I2C WRITE &H3C,0,LEN(SendData$), SendData$
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to send command byte to OLED
SUB OLEDcmd(d as INTEGER)
  I2C WRITE &H3C,0,2, &H80,d
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to send a data byte to OLED
SUB OLEDdata(d AS INTEGER)
  I2C WRITE &H3C,0,2,&H40,d
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to clear a page of data
SUB Pageclear
  OLEDcmd(&H02)  ' set column start address lower nibble
  OLEDcmd(&H10)  ' and higher nibble
  PStr$(0)=string$(128,0)
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to prepare a page/line of 16 stars
SUB AssemblePage2
  PStr$(0)="" ' clear string for a clean start
  PStr$(0)=PStr$(0)+CHR$(&H08)+CHR$(&H2A)+CHR$(&H1C)+CHR$(&H7F)+CHR$(&H1C)
  PStr$(0)=PStr$(0)+CHR$(&H2A)+CHR$(&H08)+CHR$(&H00) ' one star
  PStr$(0)=PStr$(0)+PStr$(0)+PStr$(0)+PStr$(0)
  PStr$(0)=PStr$(0)+PStr$(0)+PStr$(0)+PStr$(0)
END SUB
  ' --------------------------------------------------------------------
  ' subroutine to prepare a page/line with a star at each end
SUB AssemblePage3
  PStr$(0)="" ' clear string for a clean start
  PStr$(0)=PStr$(0)+CHR$(&H08)+CHR$(&H2A)+CHR$(&H1C)+CHR$(&H7F)+CHR$(&H1C)
  PStr$(0)=PStr$(0)+CHR$(&H2A)+CHR$(&H08)+CHR$(&H00) ' one star
  PStr$(0)=PStr$(0)+String$(112,0)+PStr$(0)
END SUB
  ' --------------------------------------------------------------------
  ' subroutine to prepare a page/line with a star at each end plus
  ' text in the centre
SUB AssemblePage5
  PStr$(0)="" ' clear string for a clean start
  PStr$(0)=PStr$(0)+CHR$(&H08)+CHR$(&H2A)+CHR$(&H1C)+CHR$(&H7F)+CHR$(&H1C)
  PStr$(0)=PStr$(0)+CHR$(&H2A)+CHR$(&H08)+CHR$(&H00) ' one star
  TStr$=CHR$(&H08)+CHR$(&H7E)+CHR$(&H09)+CHR$(&H09)+CHR$(&H00) 'f
  TStr$=TStr$+CHR$(&H7C)+CHR$(&H08)+CHR$(&H04)+CHR$(&H08)+CHR$(&H00) 'r
  TStr$=TStr$+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H38)+CHR$(&H00) 'o
  TStr$=TStr$+CHR$(&H78)+CHR$(&H04)+CHR$(&H78)+CHR$(&H04)+CHR$(&H78)+CHR$(&H00) 'm
  TStr$=TStr$+STRING$(3,0) ' space
  TStr$=TStr$+CHR$(&H7E)+CHR$(&H01)+CHR$(&H7E)+CHR$(&H01)+CHR$(&H7E)+CHR$(&H00) 'M
  TStr$=TStr$+CHR$(&H00)+CHR$(&H7D)+CHR$(&H40)+CHR$(&H00) 'i
  TStr$=TStr$+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H08)+CHR$(&H00) 'c
  TStr$=TStr$+CHR$(&H7C)+CHR$(&H08)+CHR$(&H04)+CHR$(&H08)+CHR$(&H00) 'r
  TStr$=TStr$+CHR$(&H38)+CHR$(&H44)+CHR$(&H44)+CHR$(&H38)+CHR$(&H00) 'o  
  TStr$=TStr$+CHR$(&H78)+CHR$(&H04)+CHR$(&H78)+CHR$(&H04)+CHR$(&H78)+CHR$(&H00) 'm
  TStr$=TStr$+CHR$(&H00)+CHR$(&H7D)+CHR$(&H40)+CHR$(&H00) 'i
  TStr$=TStr$+CHR$(&H04)+CHR$(&H3E)+CHR$(&H44)+CHR$(&H44)+CHR$(&H40)+CHR$(&H00) 't
  TStr$=TStr$+CHR$(&H38)+CHR$(&H54)+CHR$(&H54)+CHR$(&H18)+CHR$(&H00) 'e
  TStr$=TStr$+CHR$(&H00)+CHR$(&H5F)+CHR$(&H00) '!
  PStr$(0)=PStr$(0)+STRING$(19,0)+TStr$+STRING$(20,0)+PStr$(0)
END SUB
  ' ----------------------------------------------------------------------------
  ' subroutine to set the contrast (brightness) level of the OLED
  ' (Note that after a reset, the brightness is set at 7Fh = midway)
SUB ContrastSet
  I2C WRITE &H3C,0,2, &H80,&H81   ' enable contrast control mode
  I2C WRITE &H3C,0,2, &H80,&H9F   ' set level a bit higher than 7Fh
END SUB  
  
  ' ******************************************************************